const axios = require('axios').default
const qs = require('qs')
require('dotenv').config()
const fs = require('fs/promises')

const inquireUrl = process.env.INQUIRE_URL
const inquireUsername = process.env.INQUIRE_USERNAME
const inquirePassword = process.env.INQUIRE_PASSWORD
const lds = process.env.INQUIRE_LDS
const inquireQueryTimeout = process.env.INQUIRE_QUERY_TIMEOUT
const listAllSessionIds = 'la s.id : s.endTime == "now-1h/h"';
const sessionIdField = "s.id";

const chunkArray = (array, chunkSize) => {
    if (chunkSize === 0 || array.length === 0)
        return []
    const chunks = []
    for (let i = 0; i < array.length; i += chunkSize) {
        const chunk = array.slice(i, i + chunkSize);
        if (chunk.length > 0) {
            chunks.push(chunk)
        }
    }
    return chunks
}

const login = async () => {
    const response = await axios.post('/auth/login', {
        "user": inquireUsername, "pass": inquirePassword
    })
    return response.data
}

const logout = async () => {
    await axios.post('/auth/logout', {}, {
        headers: {
            'Accept': 'application/json'
        }
    })
}

const listLdss = async () => {
    const response = await axios.get('/data/ldss')
    return response.data
}

const submitQuery = async (query) => {
    const response = await axios.post('/tql/submit', qs.stringify({"query": query}), {
        params: {
            lds: lds,
            timeout: inquireQueryTimeout
        },
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded'
        }
    })
    return response.data
}

const waitForQuery = async (id) => {
    let response = await axios.get('/tql/poll', {params: {id: id}})
    while (response.data.type === "partial") { // Using while loop instead of recursion in case of very long-running queries
        response = await axios.get('/tql/poll', {params: {id: id}})
    }
    if (response.data.type === "final") {
        return response.data.result
    } else if (response.data.type === "failure") {
        throw new Error(response.data.error)
    } else if (response.data.type === "cancelled") {
        throw new Error("Query cancelled")
    }
}

const runQuery = async (query) => {
    const response = await submitQuery(query)
    if (response.type === "immediate") {
        return response.result
    } else {
        return await waitForQuery(response.id)
    }
}

const getSessions = async (sessionIds) => {
    const response = await axios.get(`/data/${lds}/session`,
        {
            params: {id: sessionIds},
            paramsSerializer: (params) => qs.stringify(params, {arrayFormat: 'repeat'})
        }
    )
    return response.data
}

const importSession = async (session) => {
    axios.post(``)
}

try {
    (async () => {
        axios.defaults.baseURL = inquireUrl
        axios.defaults.headers.common['Accept'] = 'application/json'
        try {
            const token = await login()
            axios.defaults.headers.common['Authorization'] = 'Bearer ' + token
        } catch (e) {
            console.log("login failed")
            console.error(e)
            return
        }

        try {
            const res = await runQuery(listAllSessionIds)

            console.log(`Found ${res.length} sessions ${JSON.stringify(res.map(row => row[sessionIdField]))}`)
            const chunkedSessionIds = chunkArray(res.map(row => row[sessionIdField]), 10)
            for (const sessionIds of chunkedSessionIds) {
                const sessionData = await getSessions(sessionIds)

                for (const sessionId of Object.keys(sessionData)) {
                    await fs.writeFile(`${sessionId}.json`, JSON.stringify(sessionData[sessionId], null, 2))
                    console.log(`Wrote session ${sessionId}`)
                }
            }
        } catch (e) {
            console.error(e.message)
        } finally {
            await logout()
        }
    })()
} catch (error) {
    console.error(error.message)
}